/*************************************************************************************
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright ? 2007 Atheros Communications, Inc.,  All Rights Reserved.
 ************************************************************************************/

/*************************************************************************************
 * Manage the atheros ethernet PHY.
 *
 * All definitions in this file are operating system independent!
 ************************************************************************************/
#include <linux/config.h>
#include <linux/types.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/delay.h>
#include "ar531xlnx.h"
#include "ae531xmac.h"
#include "ae531xreg.h"
#include "athrs26_phy.h"

void athr_VLANInit(void);
void athrs26_reg_init(void);
BOOL athrs26_igmps_enable(UINT32);





/* PHY selections and access functions */

typedef enum {
    PHY_SRCPORT_INFO, 
    PHY_PORTINFO_SIZE,
} PHY_CAP_TYPE;

typedef enum {
    PHY_SRCPORT_NONE,
    PHY_SRCPORT_VLANTAG, 
    PHY_SRCPORT_TRAILER,
} PHY_SRCPORT_TYPE;

#define DRV_LOG(DBG_SW, X0, X1, X2, X3, X4, X5, X6)
#define DRV_MSG(x,a,b,c,d,e,f)
#define DRV_PRINT(DBG_SW,X)

#define ATHR_LAN_PORT_VLAN          1
#define ATHR_WAN_PORT_VLAN          2

#define ENET_UNIT_DEFAULT 0

#define TRUE    1
#define FALSE   0

#define ATHR_PHY0_ADDR   0x0
#define ATHR_PHY1_ADDR   0x1
#define ATHR_PHY2_ADDR   0x2
#define ATHR_PHY3_ADDR   0x3
#define ATHR_PHY4_ADDR   0x4

/*****************************igmps*******************************/
#define PORT_CTL_OFFSET  0x0104
#define PORT_CTL_E_OFFSET         0x0100
#define PORT_CTL_IGMP_MLD_EN_BOFFSET 10
/*****************************Device ID***************************/
#define DEVICE_ID_OFFSET 0x0
#define LOAD_EEPROM_E_BOFFSET 16
/*****************************IOCTL*******************************/
#define ATHRCGREG 0x101
#define ATHRCSREG 0x102
#define ATHRCGETY 0x103
#define ATHRCSETY 0x104
#define ATHRCGFLD 0x105
#define ATHRCSFLD 0x106

#ifndef DEBUG_CMD
#define DEBUG_CMD 
#endif

#ifdef DEBUG_CMD
#include <asm/uaccess.h>
#endif
/*****************************VLAN*******************************/
#if VLANConfig
 #ifndef HEADER_EN
 #define HEADER_EN
 #endif
#else
 #ifdef HEADER_EN
 #undef HEADER_EN
 #endif
#endif


/*
 * Track per-PHY port information.
 */
typedef struct {
    BOOL   isEnetPort;       /* normal enet port */
    BOOL   isPhyAlive;       /* last known state of link */
    int    ethUnit;          /* MAC associated with this phy port */
    UINT32 phyBase;
    UINT32 phyAddr;          /* PHY registers associated with this phy port */
    UINT32 VLANTableSetting; /* Value to be written to VLAN table */
} athrPhyInfo_t;

/*
 * Per-PHY information, indexed by PHY unit number.
 */
static athrPhyInfo_t athrPhyInfo[] = {
    {TRUE,   /* phy port 1 -- LAN port 0 */
     FALSE,
     ENET_UNIT_DEFAULT,
     (UINT32) (PHYS_TO_K1(AR531X_ENET0)+AE531X_PHY_OFFSET),
     ATHR_PHY0_ADDR,
     ATHR_LAN_PORT_VLAN
    },

    {TRUE,   /* phy port 2 -- LAN port 1 */
     FALSE,
     ENET_UNIT_DEFAULT,
     (UINT32) (PHYS_TO_K1(AR531X_ENET0)+AE531X_PHY_OFFSET),
     ATHR_PHY1_ADDR,
     ATHR_LAN_PORT_VLAN
    },

    {TRUE,   /* phy port 3 -- LAN port 2 */
     FALSE,
     ENET_UNIT_DEFAULT,
     (UINT32) (PHYS_TO_K1(AR531X_ENET0)+AE531X_PHY_OFFSET),
     ATHR_PHY2_ADDR, 
     ATHR_LAN_PORT_VLAN
    },

    {TRUE,   /* phy port 4 -- LAN port 3 */
     FALSE,
     ENET_UNIT_DEFAULT,
     (UINT32) (PHYS_TO_K1(AR531X_ENET0)+AE531X_PHY_OFFSET),
     ATHR_PHY3_ADDR, 
     ATHR_LAN_PORT_VLAN
    },

    {TRUE,   /* phy port 5 -- WAN port or LAN port 4 */
     FALSE,
     ENET_UNIT_DEFAULT,
     (UINT32) (PHYS_TO_K1(AR531X_ENET0)+AE531X_PHY_OFFSET),
     ATHR_PHY4_ADDR, 
     ATHR_LAN_PORT_VLAN   /* Send to all ports */
    },
    
    {FALSE,  /* phy port 0 -- CPU port (no RJ45 connector) */
     TRUE,
     ENET_UNIT_DEFAULT,
     (UINT32) (PHYS_TO_K1(AR531X_ENET0)+AE531X_PHY_OFFSET),
     0x00, 
     ATHR_LAN_PORT_VLAN    /* Send to all ports */
    },
};

static UINT8 athr26_init_flag = 0;

#define ATHR_GLOBALREGBASE    0

#define ATHR_PHY_MAX (sizeof(athrPhyInfo) / sizeof(athrPhyInfo[0]))

/* Range of valid PHY IDs is [MIN..MAX] */
#define ATHR_ID_MIN 0
#define ATHR_ID_MAX (ATHR_PHY_MAX-1)

/* Convenience macros to access myPhyInfo */
#define ATHR_IS_ENET_PORT(phyUnit) (athrPhyInfo[phyUnit].isEnetPort)
#define ATHR_IS_PHY_ALIVE(phyUnit) (athrPhyInfo[phyUnit].isPhyAlive)
#define ATHR_ETHUNIT(phyUnit) (athrPhyInfo[phyUnit].ethUnit)
#define ATHR_PHYBASE(phyUnit) (athrPhyInfo[phyUnit].phyBase)
#define ATHR_PHYADDR(phyUnit) (athrPhyInfo[phyUnit].phyAddr)
#define ATHR_VLAN_TABLE_SETTING(phyUnit) (athrPhyInfo[phyUnit].VLANTableSetting)

/* VLAN */
#define LAN_PORT(x)	(x < 4)
#define WAN_PORT(x)	(x == 4)

#define ATHR_IS_ETHUNIT(phyUnit, ethUnit) \
            (ATHR_IS_ENET_PORT(phyUnit) &&        \
            ATHR_ETHUNIT(phyUnit) == (ethUnit))

#define ATHR_IS_WAN_PORT(phyUnit) (!(ATHR_ETHUNIT(phyUnit)==ENET_UNIT_DEFAULT))
            
/* Forward references */
BOOL       athrs26_phy_is_link_alive(int phyUnit);
static UINT32 athrs26_reg_read(UINT32 reg_addr);
static void athrs26_reg_write(UINT32 reg_addr, UINT32 reg_val);

/******************************************************************************
*
* athrs26_phy_is_link_alive - test to see if the specified link is alive
*
* RETURNS:
*    TRUE  --> link is alive
*    FALSE --> link is down
*/
BOOL
athrs26_phy_is_link_alive(int phyUnit)
{
    UINT16 phyHwStatus;
    UINT32 phyBase;
    UINT32 phyAddr;

    phyBase = ATHR_PHYBASE(phyUnit);
    phyAddr = ATHR_PHYADDR(phyUnit);

    phyHwStatus = phyRegRead(phyBase, phyAddr, ATHR_PHY_SPEC_STATUS);

    if (phyHwStatus & ATHR_STATUS_LINK_PASS)
        return TRUE;

    return FALSE;
}

void athrs26_reg_init()
{
    int port;
    
    /* if using header for register configuration, we have to     */
    /* configure s26 register after frame transmission is enabled */
    if (athr26_init_flag)
        return;
    
    athrs26_reg_write(0x200, 0x200);
    athrs26_reg_write(0x300, 0x200);
    athrs26_reg_write(0x400, 0x200);
    athrs26_reg_write(0x500, 0x200);
    athrs26_reg_write(0x600, 0x200);
       
    athrs26_reg_write(0x38, 0xc000050e);
#ifdef HEADER_EN        
    athrs26_reg_write(0x104, 0x6804);
#else
    athrs26_reg_write(0x104, 0x6004);
#endif
       
    athrs26_reg_write(0x60, 0xffffffff);
    athrs26_reg_write(0x64, 0xaaaaaaaa);
    athrs26_reg_write(0x68, 0x55555555);    
    athrs26_reg_write(0x6c, 0x0);    
    athrs26_reg_write(0x70, 0x41af);

#ifdef FULL_FEATURE
    athena_init(0, 1);
#endif
   
#if VLANConfig 
    /* default vlan setting, just separate the LAN and WAN */
    athr_VLANInit();
#endif

#if 0	
    for (port = 1; port <= 5; port ++ ) {
        /* enable IGMP */
        athrs26_igmps_enable(port);
    }
#endif
    
    athr26_init_flag = 1;
}

/******************************************************************************
*
* athrs26_phy_setup - reset and setup the PHY associated with
* the specified MAC unit number.
*
* Resets the associated PHY port.
*
* RETURNS:
*    TRUE  --> associated PHY is alive
*    FALSE --> no LINKs on this ethernet unit
*/

BOOL
athrs26_phy_setup(int ethUnit, UINT32 _phyBase)
{
    int     phyUnit;
    UINT16  phyHwStatus;
    UINT16  timeout;
    int     liveLinks = 0;
    UINT32  phyBase = 0;
    BOOL    foundPhy = FALSE;
    UINT32  phyAddr = 0;

    /* See if there's any configuration data for this enet */
    /* start auto negogiation on each phy */
    printk("-------------AR8216 Ver:%s---------------\n",VERSION);
    for (phyUnit=0; phyUnit < ATHR_PHY_MAX; phyUnit++) {
        if (!ATHR_IS_ETHUNIT(phyUnit, ethUnit)) {
            continue;
        }
        
        foundPhy = TRUE;
        phyBase = ATHR_PHYBASE(phyUnit);
        phyAddr = ATHR_PHYADDR(phyUnit);
        
        phyRegWrite(phyBase, phyAddr, ATHR_AUTONEG_ADVERT,
                      ATHR_ADVERTISE_ALL);

        /* Reset PHYs*/
        phyRegWrite(phyBase, phyAddr, ATHR_PHY_CONTROL,
                      ATHR_CTRL_AUTONEGOTIATION_ENABLE 
                      | ATHR_CTRL_SOFTWARE_RESET);
    }

    if (!foundPhy) {
        return FALSE; /* No PHY's configured for this ethUnit */
    }
    
    /*
     * After the phy is reset, it takes a little while before
     * it can respond properly.
     */
    sysMsDelay(2000);
    
    /*
     * Wait up to 3 seconds for ALL associated PHYs to finish
     * autonegotiation.  The only way we get out of here sooner is
     * if ALL PHYs are connected AND finish autonegotiation.
     */
    for (phyUnit=0; (phyUnit < ATHR_PHY_MAX) /*&& (timeout > 0) */; phyUnit++) {
        if (!ATHR_IS_ETHUNIT(phyUnit, ethUnit)) {
            continue;
        }

        timeout=20;
        for (;;) {
            phyHwStatus = phyRegRead(phyBase, phyAddr, ATHR_PHY_CONTROL);

            if (ATHR_RESET_DONE(phyHwStatus)) {
                DRV_PRINT(DRV_DEBUG_PHYSETUP,
                          ("Port %d, Neg Success\n", phyUnit));
                break;
            }
            if (timeout == 0) {
                DRV_PRINT(DRV_DEBUG_PHYSETUP,
                          ("Port %d, Negogiation timeout\n", phyUnit));
                break;
            }
            if (--timeout == 0) {
                DRV_PRINT(DRV_DEBUG_PHYSETUP,
                          ("Port %d, Negogiation timeout\n", phyUnit));
                break;
            }

            sysMsDelay(150);
        }
#ifdef S26_VER_1_0
        //turn off power saving
        phyRegWrite(ATHR_PHYBASE(0), phyUnit, 29, 41);
        phyRegWrite(ATHR_PHYBASE(0), phyUnit, 30, 0);
#endif
    }

    /*
     * All PHYs have had adequate time to autonegotiate.
     * Now initialize software status.
     *
     * It's possible that some ports may take a bit longer
     * to autonegotiate; but we can't wait forever.  They'll
     * get noticed by mv_phyCheckStatusChange during regular
     * polling activities.
     */
    for (phyUnit=0; phyUnit < ATHR_PHY_MAX; phyUnit++) {
        if (!ATHR_IS_ETHUNIT(phyUnit, ethUnit)) {
            continue;
        }

        if (athrs26_phy_is_link_alive(phyUnit)) {
            liveLinks++;
            ATHR_IS_PHY_ALIVE(phyUnit) = TRUE;
        } else {
            ATHR_IS_PHY_ALIVE(phyUnit) = FALSE;
        }

        DRV_PRINT(DRV_DEBUG_PHYSETUP,
            ("eth%d: Phy Specific Status=%4.4x\n",
            ethUnit, 
            phyRegRead(ATHR_PHYBASE(phyUnit),
                         ATHR_PHYADDR(phyUnit),
                         ATHR_PHY_SPEC_STATUS)));
    }


    /* if using header for register configuration, we have to     */
    /* configure s26 register after frame transmission is enabled */
    athrs26_reg_init();

    return (liveLinks > 0);
}

/******************************************************************************
*
* athrs26_phy_is_fdx - Determines whether the phy ports associated with the
* specified device are FULL or HALF duplex.
*
* RETURNS:
*    1 --> FULL
*    0 --> HALF
*/
int
athrs26_phy_is_fdx(int ethUnit)
{
    int       phyUnit;
    UINT32  phyBase;
    UINT32  phyAddr;
    UINT16  phyHwStatus;
    int       ii = 200;

    if (ethUnit == ENET_UNIT_DEFAULT)
        return TRUE;

    for (phyUnit=0; phyUnit < ATHR_PHY_MAX; phyUnit++) {
        if (!ATHR_IS_ETHUNIT(phyUnit, ethUnit)) {
            continue;
        }

        if (athrs26_phy_is_link_alive(phyUnit)) {

            phyBase = ATHR_PHYBASE(phyUnit);
            phyAddr = ATHR_PHYADDR(phyUnit);

            do {
                phyHwStatus = phyRegRead(phyBase, phyAddr, 
                                               ATHR_PHY_SPEC_STATUS);
                sysMsDelay(10);
            } while((!(phyHwStatus & ATHR_STATUS_RESOVLED)) && --ii);
            
            if (phyHwStatus & ATHER_STATUS_FULL_DEPLEX)
                return TRUE;
        }
    }

    return FALSE;
}


/******************************************************************************
*
* athrs26_phy_speed - Determines the speed of phy ports associated with the
* specified device.
*
* RETURNS:
*               AG7100_PHY_SPEED_10T, AG7100_PHY_SPEED_100TX;
*               AG7100_PHY_SPEED_1000T;
*/

BOOL 
athrs26_phy_speed(int ethUnit)
{
    int       phyUnit;
    UINT16  phyHwStatus;
    UINT32  phyBase;
    UINT32  phyAddr;
    int       ii = 200;

    for (phyUnit=0; phyUnit < ATHR_PHY_MAX; phyUnit++) {
        if (!ATHR_IS_ETHUNIT(phyUnit, ethUnit)) {
            continue;
        }

        if (athrs26_phy_is_link_alive(phyUnit)) {

            phyBase = ATHR_PHYBASE(phyUnit);
            phyAddr = ATHR_PHYADDR(phyUnit);
            do {
                phyHwStatus = phyRegRead(phyBase, phyAddr, 
                                              ATHR_PHY_SPEC_STATUS);
                sysMsDelay(10);
            }while((!(phyHwStatus & ATHR_STATUS_RESOVLED)) && --ii);
            
            phyHwStatus = ((phyHwStatus & ATHER_STATUS_LINK_MASK) >>
                           ATHER_STATUS_LINK_SHIFT);

            switch(phyHwStatus) {
            case 0:
                return FALSE;
            case 1:
                return TRUE;
            default:
                printk("Unkown speed read!\n");
            }
        }
    }
    return FALSE;
}

/*****************************************************************************
*
* athrs26_phyCheckStatusChange -- checks for significant changes in PHY state.
*
* A "significant change" is:
*     dropped link (e.g. ethernet cable unplugged) OR
*     autonegotiation completed + link (e.g. ethernet cable plugged in)
*
* When a PHY is plugged in, phyLinkGained is called.
* When a PHY is unplugged, phyLinkLost is called.
*/

void
athrs26_phyCheckStatusChange(int ethUnit)
{
    int           phyUnit;
    UINT16      phyHwStatus,phyHwControl;
    athrPhyInfo_t *lastStatus;
    int           linkCount   = 0;
    int           lostLinks   = 0;
    int           gainedLinks = 0;
    UINT32      phyBase;
    UINT32      phyAddr;

    for (phyUnit=0; phyUnit < ATHR_PHY_MAX; phyUnit++) {
        if (!ATHR_IS_ETHUNIT(phyUnit, ethUnit)) {
            continue;
        }

        phyBase = ATHR_PHYBASE(phyUnit);
        phyAddr = ATHR_PHYADDR(phyUnit);

        lastStatus = &athrPhyInfo[phyUnit];
        phyHwStatus = phyRegRead(phyBase, phyAddr, ATHR_PHY_SPEC_STATUS);

        if (lastStatus->isPhyAlive) { /* last known link status was ALIVE */
            /* See if we've lost link */
            if (phyHwStatus & ATHR_STATUS_LINK_PASS) {
                linkCount++;
            } else {
                lostLinks++;
                DRV_PRINT(DRV_DEBUG_PHYCHANGE,("\nenet%d port%d down\n",
                                               ethUnit, phyUnit));
                lastStatus->isPhyAlive = FALSE;
            }
        } else { /* last known link status was DEAD */
            /* Check for reset complete */
            phyHwStatus = phyRegRead(phyBase, phyAddr, ATHR_PHY_STATUS);
            if (!ATHR_RESET_DONE(phyHwStatus))
                continue;

            phyHwControl = phyRegRead(phyBase, phyAddr, ATHR_PHY_CONTROL);
            /* Check for AutoNegotiation complete */            
            if ((!(phyHwControl & ATHR_CTRL_AUTONEGOTIATION_ENABLE)) 
                 || ATHR_AUTONEG_DONE(phyHwStatus)) {
                phyHwStatus = phyRegRead(phyBase, phyAddr, 
                                         ATHR_PHY_SPEC_STATUS);

                if (phyHwStatus & ATHR_STATUS_LINK_PASS) {
                    gainedLinks++;
                    linkCount++;
                    DRV_PRINT(DRV_DEBUG_PHYCHANGE,("\nenet%d port%d up\n",
                                                   ethUnit, phyUnit));
                    lastStatus->isPhyAlive = TRUE;
                }
            }
        }
    }

    if (linkCount == 0) {
        if (lostLinks) {
            /* We just lost the last link for this MAC */
            phyLinkLost(ethUnit);
        }
    } else {
        if (gainedLinks == linkCount) {
            /* We just gained our first link(s) for this MAC */
            phyLinkGained(ethUnit);
        }
    }
}

static UINT32
athrs26_reg_read(UINT32 reg_addr)
{
    UINT32 reg_word_addr;
    UINT32 phy_addr, tmp_val, reg_val;
    UINT16 phy_val;
    UINT8  phy_reg; 

    /* read the first 16 bits*/
    reg_word_addr = (reg_addr & 0xfffffffc) >> 1;
    /* configure register high address */
    phy_addr = 0x18;
    phy_reg = 0x0;
    phy_val = (reg_word_addr >> 8) & 0x1ff;         /* bit16-8 of reg address*/
    phyRegWrite (ATHR_PHYBASE(0), phy_addr, phy_reg, phy_val);

    /* read register with low address */
    phy_addr = 0x10 | ((reg_word_addr >> 5) & 0x7); /* bit7-5 of reg address */
    phy_reg = (UINT8) (reg_word_addr & 0x1f);   /* bit4-0 of reg address */
    reg_val = phyRegRead(ATHR_PHYBASE(0), phy_addr, phy_reg);

    /* read the second 16 bits*/
    reg_word_addr++;

    phy_addr = 0x10 | ((reg_word_addr >> 5) & 0x7); /* bit7-5 of reg address */
    phy_reg = (UINT8)(reg_word_addr & 0x1f);                 /* bit4-0 of reg address */
	tmp_val = (UINT32)phyRegRead(ATHR_PHYBASE(0), phy_addr, phy_reg);
    reg_val |= (tmp_val << 16);
    return reg_val;
}

static void
athrs26_reg_write(UINT32 reg_addr, UINT32 reg_val)
{
    UINT32 reg_word_addr;
    UINT32 phy_addr;
    UINT16 phy_val;
    UINT8  phy_reg; 

    /* change reg_addr to 16-bit word address, 32-bit aligned */
    reg_word_addr = (reg_addr & 0xfffffffc) >> 1;
    /* configure register high address */
    phy_addr = 0x18;
    phy_reg = 0x0;
    phy_val = (UINT16)(reg_word_addr >> 8) & 0x1ff;         /* bit16-8 of reg address*/
    phyRegWrite(ATHR_PHYBASE(0), phy_addr, phy_reg, phy_val);

    /* For some registers such as ARL and VLAN, since they include BUSY bit */
    /* in lower address, we should write the higher 16-bit register then the */
    /* lower one */
    
    /* read register in higher address */
    reg_word_addr++;    
    phy_addr = 0x10 | ((reg_word_addr >> 5) & 0x7); /* bit7-5 of reg address */
    phy_reg = (UINT8) (reg_word_addr & 0x1f);   /* bit4-0 of reg address */
    phy_val = (UINT16) ((reg_val >> 16) & 0xffff);
    phyRegWrite(ATHR_PHYBASE(0), phy_addr, phy_reg, phy_val);

    /* write register in lower address */
    reg_word_addr--;
    phy_addr = 0x10 | ((reg_word_addr >> 5) & 0x7); /* bit7-5 of reg address */
    phy_reg = (UINT8) (reg_word_addr & 0x1f);   /* bit4-0 of reg address */
    phy_val = (UINT16) (reg_val & 0xffff);
    phyRegWrite(ATHR_PHYBASE(0), phy_addr, phy_reg, phy_val); 
}
int
athr_ioctl(UINT32 unit, UINT32 *args)
{
#ifdef DEBUG_CMD
    switch (args[0]) {
    case ATHRCGREG: {
        UINT32 reg_val;
        
        reg_val = (UINT32)athrs26_reg_read(args[1]);
        if (copy_to_user(args[2], &reg_val, 
            sizeof(UINT32)))
          return EFAULT;

        return 0;
    }

    case ATHRCSREG: {
        athrs26_reg_write ((UINT16)args[1], (UINT32)args[2]);
        return 0;
    }
#if 0
    case ATHRCGETY: {
        UINT32 reg_addr, cmd_num, index;
        UINT16 entry_len;
        UINT8 reg_tmp[4];
        UINT32 reg_val;
        ebu_hsl_reg_t *p_reg = (ebu_hsl_reg_t *)athena_reg_ptr_get();
        EBU_STATUS rtn;

        if (!p_reg)
            return ENXIO;
        
        cmd_num = args[1];
        index = args[2];  

        reg_addr = reg_desc[cmd_num].reg_addr;
        entry_len = 4;
        
        rtn = p_reg->reg_entry_get (0, reg_addr, 
                                    index * reg_desc[cmd_num].entry_offset,
                                    entry_len, reg_tmp); 
        
        if (rtn == EBU_ERROR)
            return EINVAL;

        reg_val = ((UINT32)((reg_tmp[0] << 24) & 0xff000000)) | 
                  ((UINT32)((reg_tmp[1] << 16) & 0x00ff0000)) |
                  ((UINT32)((reg_tmp[2] << 8) & 0x0000ff00)) | 
                  ((UINT32)reg_tmp[3]) ;

      if (copy_to_user(args[3], &reg_val,
            sizeof(UINT32)))
        return EFAULT;

        return 0;
    }
  
    case ATHRCSETY: {
        UINT32 reg_addr, cmd_num, index;
        UINT16 entry_len;
        UINT8 reg_tmp[4];
        ebu_hsl_reg_t *p_reg = (ebu_hsl_reg_t *)athena_reg_ptr_get(); 
        EBU_STATUS rtn;
       
        if (!p_reg)
            return ENXIO;
        
        cmd_num = args[1];
        index = args[2];
    
        reg_addr = reg_desc[cmd_num].reg_addr;
        entry_len = 4;
        reg_tmp[0] = (UINT8)((args[3] & 0xff000000) >> 24);//reg address high
        reg_tmp[1] = (UINT8)((args[3] & 0x00ff0000) >> 16);
        reg_tmp[2] = (UINT8)((args[3] & 0x0000ff00) >> 8);//reg address low
        reg_tmp[3] = (UINT8)(args[3] & 0x000000ff);
    
        rtn = p_reg->reg_entry_set (0, reg_addr, (UINT16)
                                    (index * reg_desc[cmd_num].entry_offset),
                                    entry_len, reg_tmp);
        if (rtn == EBU_ERROR)
            return EINVAL;
        
        return 0;
    }
        
    case ATHRCGFLD: {
        UINT32 reg_addr, cmd_num, index;
        UINT16 entry_offset, field_offset, field_len;
        UINT32 field_val;
        UINT8 field_tmp[4];
        ebu_hsl_reg_t *p_reg = (ebu_hsl_reg_t *)athena_reg_ptr_get(); 
        EBU_STATUS rtn;

        if (!p_reg)
            return ENXIO;
        
        cmd_num = args[1];
        index = args[2];  

        memset(field_tmp, 0, sizeof(field_tmp));
        reg_addr = reg_desc[field_desc[cmd_num].reg_id].reg_addr;
        field_len = field_desc[cmd_num].bit_len;
        field_offset = field_desc[cmd_num].bit_offset;
        entry_offset = index * 
                       reg_desc[field_desc[cmd_num].reg_id].entry_offset;        
        rtn = p_reg->reg_field_get (0, reg_addr, entry_offset,
                                    field_offset, field_len, field_tmp);
        
        if (rtn == EBU_ERROR)
            return EINVAL;

        switch ((field_len-1) / 8) {
        case 0:
            field_val = field_tmp[0];
            break;
            
        case 1:
            field_val = ((UINT32)((field_tmp[0] << 8) & 0xff00)) | 
                        ((UINT32)field_tmp[1]);
            break;
            
        case 2:
            field_val = ((UINT32)((field_tmp[0] << 16) & 0xff0000)) | 
                        ((UINT32)((field_tmp[1] << 8) & 0xff00)) |                 
                        ((UINT32)field_tmp[2]);
            break;
            
        case 3:
            field_val = ((UINT32)((field_tmp[0] << 24) & 0xff000000)) | 
                        ((UINT32)((field_tmp[1] << 16) & 0xff0000)) |                 
                        ((UINT32)((field_tmp[2] << 8) & 0xff00)) |                 
                        ((UINT32)field_tmp[3]);
            break;
            
        default:
            return EINVAL;
        }
        
      if (copy_to_user(args[3], &field_val,
                       sizeof(UINT32)))
        return EFAULT;

        return 0;          
    }
  
    case ATHRCSFLD:
    {
        UINT32 reg_addr, cmd_num, index;
        UINT16 entry_offset, field_offset, field_len;
        UINT8 field_tmp[4];
        ebu_hsl_reg_t *p_reg = (ebu_hsl_reg_t *)athena_reg_ptr_get(); 
        EBU_STATUS rtn;

        if (!p_reg)
            return ENXIO;
        
        cmd_num = args[1];
        index = args[2];  

        reg_addr = reg_desc[field_desc[cmd_num].reg_id].reg_addr;
        field_len = field_desc[cmd_num].bit_len;
        field_offset = field_desc[cmd_num].bit_offset;

        entry_offset = index * 
                       reg_desc[field_desc[cmd_num].reg_id].entry_offset;
        memset(field_tmp, 0, sizeof(field_tmp));
        
        switch ((field_len-1) / 8) {
        case 0:
            field_tmp[0] = (UINT8)args[3];
            break;
            
        case 1:
            field_tmp[0] = (UINT8)((0xff00 & args[3]) >> 8);
            field_tmp[1] = (UINT8)(0x00ff & args[3]);     
            break;
            
        case 2:
            field_tmp[0] = (UINT8)((0xff0000 & args[3]) >> 16);
            field_tmp[1] = (UINT8)((0xff00 & args[3]) >> 8);      
            field_tmp[2] = (UINT8)(0x00ff & args[3]);   
            break;
            
        case 3:
            field_tmp[0] = (UINT8)((0xff0000 & args[3]) >> 24);
            field_tmp[1] = (UINT8)((0xff0000 & args[3]) >> 16);
            field_tmp[2] = (UINT8)((0xff00 & args[3]) >> 8);      
            field_tmp[3] = (UINT8)(0x00ff & args[3]); 

            break;
            
        default:
            return EBU_ERROR;
        }

        rtn = p_reg->reg_field_set (0, reg_addr, entry_offset,
                                    field_offset, field_len, field_tmp);
        
        if (rtn == EBU_ERROR)
            return EINVAL;
    
        return 0;
    }
#endif
    default:
        return -EOPNOTSUPP;
    }
#endif /* DEBUG_CMD*/
    //KW
    //return -EOPNOTSUPP;
    return TRUE;
}

void 
athr_VLANInit()
{
    //Create Vlan 1, add member port 0, 1, 2, 3, 4.
    printk("enter athr_VLANInit\n");
    athrs26_reg_write(0x44, 0x0000081f);   
    athrs26_reg_write(0x40, 0x0001000a);

    
    //Create Vlan 2, add member port 0,5.
    athrs26_reg_write(0x44, 0x00000821);  
    athrs26_reg_write(0x40, 0x0002000a);

    //Configure 802.1Q_MODE, PORT_VID_MEM, PORT_VID

    athrs26_reg_write(0x104, 0x6a04);	  //egress frames with vlan tag  0x4004
    athrs26_reg_write(0x108, 0xc3fe0001);  //port 0 vlan 1 & vlan member all

    athrs26_reg_write(0x204, 0x6104);     
    athrs26_reg_write(0x208, 0xc01d0001);

    athrs26_reg_write(0x304, 0x6104);   
    athrs26_reg_write(0x308, 0xc01b0001);
	
    athrs26_reg_write(0x404, 0x6104);      
    athrs26_reg_write(0x408, 0xc0170001);
	
    athrs26_reg_write(0x504, 0x6104);     
    athrs26_reg_write(0x508, 0xc00f0001);
	
    athrs26_reg_write(0x604, 0x6104);     
    athrs26_reg_write(0x608, 0xc0010002);
}

BOOL athrs26_get_igmps_status(UINT32 port, UINT32 *en_flag)
{
    UINT32 reg_addr;
    UINT32 reg_data;

    if(port > 5)
        return FALSE;
            
    reg_addr = PORT_CTL_OFFSET + PORT_CTL_E_OFFSET*port;
    reg_data = athrs26_reg_read(reg_addr);
    if (reg_data & (1UL << PORT_CTL_IGMP_MLD_EN_BOFFSET) )
        *en_flag = 1;
    else
        *en_flag = 0;    		
    return TRUE;
}

BOOL athrs26_igmps_enable(UINT32 port)
{
    UINT32 reg_addr;
    UINT32 reg_data;

    if(port > 5)
        return FALSE;
        
    reg_addr = PORT_CTL_OFFSET + PORT_CTL_E_OFFSET*port;
    reg_data = athrs26_reg_read(reg_addr);
    if (!(reg_data & (1UL << PORT_CTL_IGMP_MLD_EN_BOFFSET))) {
        reg_data |= (1UL << PORT_CTL_IGMP_MLD_EN_BOFFSET); 
        athrs26_reg_write(reg_addr, reg_data);
    }
    return TRUE;    
}

BOOL athrs26_igmps_disable(UINT32 port)
{
    UINT32 reg_addr;
    UINT32 reg_data;

    if(port > 5)
        return FALSE;
        
    reg_addr = PORT_CTL_OFFSET + PORT_CTL_E_OFFSET*port;
    reg_data = athrs26_reg_read(reg_addr);
    
    printk("reg_data = 0x%08X\n",reg_data);
    if (reg_data & (1UL << PORT_CTL_IGMP_MLD_EN_BOFFSET)) {
        reg_data &= ~(1UL << PORT_CTL_IGMP_MLD_EN_BOFFSET); 
        athrs26_reg_write(reg_addr, reg_data);
    }
    return TRUE;    
}
