/**************************************************************************
 *
 *  BRIEF MODULE DESCRIPTION
 *     nvram /proc interface routines.
 *
 *  Copyright 2004 IDT Inc. (rischelp@idt.com)
 *         
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 **************************************************************************
 * Together with nvram438.c, we provide /proc interface for
 * the user to set and modify the environment variables stored in NVRAM.
 **************************************************************************
 * Examples
 * 
 * #cat /proc/nvram/boot
 *    will display the boot environment variables, like netaddr0,bootaddr etc.
 * #echo "netaddr0 192.168.1.3" >/proc/nvram/boot
 *    will modify the boot environment variable netaddr0.
 * #echo "netaddr0" >/proc/nvram/boot
 *    will erase the "netaddr0" environment variable
 * #echo >/proc/nvram/boot
 *    will erase all environment variables stored in nvram
 **************************************************************************
 * Pallathu Sadik Oct 07,2003
 * Original Form
 **************************************************************************
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/proc_fs.h>
#include <linux/sched.h>
#include <asm/uaccess.h>
#include <linux/string.h>

//#define DEBUG 1
#define MODULE_VERSION "1.0"
#define MODULE_NAME "nvram"
#define BOOT_MAJOR 240
#define BUF_LEN 8192

extern void setenv (char *e, char *v, int rewrite);
extern char *getenv (char *e);
extern void unsetenv (char *e);
extern void mapenv (int (*func)(char *, char *));
extern void purgeenv(void);
extern char *getenv (char *s);

static struct proc_dir_entry *nvram, *boot;

static int boot_open(struct inode *, struct file *);
static int boot_release(struct inode *, struct file *);
static ssize_t boot_read(struct file *, char *, size_t, loff_t *);
static ssize_t boot_write(struct file *, const char *, size_t, loff_t *);

static struct file_operations boot_fops = {
  .read = boot_read, 
  .write = boot_write,
  .open = boot_open,
  .release = boot_release
};

static char msg[BUF_LEN];
static char *msg_Ptr;
const char delimiters[] = " \n";

static int printVal(char *name,char *val)
{
  strcat(msg,name);
  strcat(msg," ");
  strcat(msg,val);
  strcat(msg,"\n");
  return 0;
}

static int boot_open(struct inode *inode, struct file *file)
{
   MOD_INC_USE_COUNT;
   memset(msg,0,BUF_LEN);
   msg_Ptr = msg;
   return 0;
}
static int boot_release(struct inode *inode, struct file *file)
{
  MOD_DEC_USE_COUNT;
  return 0;
}

static ssize_t boot_read(struct file *filp,
			   char *buffer,    /* The buffer to fill with data */
			   size_t length,   /* The length of the buffer     */
			   loff_t *offset)  /* Our offset in the file       */
{
  int bytes_read = 0;
  if(msg_Ptr == msg)
    {
      mapenv(&printVal);
    }
  if (*msg_Ptr == 0){
    msg_Ptr = msg;
    return 0;
  }
  while (length && *msg_Ptr)
    {
      put_user(*(msg_Ptr++), buffer++);
      length--;
      bytes_read++;
    }
  return bytes_read;
}

static ssize_t boot_write(struct file *filp,
			    const char *buffer,
			    size_t length,
			    loff_t *offset)
{
  int bytes_written = 0;
  char *token1,*token2;

  memset(msg,0,BUF_LEN);

  while(length && *buffer)
    {
      get_user(msg[bytes_written],buffer++);
      length--;
      bytes_written++;
    }

  token1 = strsep(&msg_Ptr,delimiters);
  token2 = strsep(&msg_Ptr,delimiters);

  if((*token1 !='\0') && (*token2 != '\0'))
    {
#ifdef DEBUG
    printk(__FILE__": setting environment %s to %s\n",token1,token2);
#endif
    setenv(token1,token2,1);
    }
  else if(*token1 != '\0')
    {
#ifdef DEBUG
      printk(__FILE__": purging environment variable %s\n",token1);
#endif
      unsetenv(token1);
    }
  else
    {
#ifdef DEBUG
      printk(__FILE__": purging all environment variables\n");
#endif
      purgeenv();
    }
  
  return bytes_written;
}


static int __init init_procfs_nvram(void)
{
  int rv = 0;

  /* create directory */
  nvram = proc_mkdir(MODULE_NAME, NULL);
  if(nvram == NULL) {
#ifdef DEBUG
    printk(__FILE__": Cannot make proc entry for 'nvram'\n");
#endif
    rv = -ENOMEM;
    goto out1;
  }
#ifdef DEBUG
  printk(__FILE__": module %s %s initialised\n", MODULE_NAME, MODULE_VERSION);
#endif
  if (register_chrdev(BOOT_MAJOR,"boot", &boot_fops) <0)
    {
#ifdef DEBUG     
      printk(__FILE__": Cannot register 'boot' device.\n");
#endif
      rv = -EINVAL;
      goto out2;
    }
      
#ifdef DEBUG     
  printk(__FILE__": device 'boot' (major %d) registered\n",BOOT_MAJOR);
#endif
        
  boot = proc_mknod("boot", S_IFCHR | 0644, nvram, MKDEV(BOOT_MAJOR, 0));
  if(boot == NULL) {
#ifdef DEBUG
    printk(__FILE__": Cannot make proc entry for boot.\n");
#endif
    rv = -ENOMEM;
    goto out3;
  }

  boot->owner = THIS_MODULE;
  return 0;
  
 out3:
  unregister_chrdev(BOOT_MAJOR,"boot");
 out2:
  remove_proc_entry(MODULE_NAME, NULL);
 out1:
  return rv;
}


static void __exit cleanup_procfs_nvram(void)
{
  remove_proc_entry("boot", nvram);
  unregister_chrdev(BOOT_MAJOR,"boot");
  remove_proc_entry(MODULE_NAME, NULL);
#ifdef DEBUG
  printk("%s %s removed\n", MODULE_NAME, MODULE_VERSION);
#endif
}


module_init(init_procfs_nvram);
module_exit(cleanup_procfs_nvram);

MODULE_AUTHOR("Pallathu Sadik");
MODULE_DESCRIPTION("proc interface for nvram");
MODULE_LICENSE("GPL");

EXPORT_NO_SYMBOLS;
