/**************************************************************************
 *
 *  BRIEF MODULE DESCRIPTION
 *      EB438 specific polling driver for 16550 UART.
 *
 *  Copyright 2004 IDT Inc. (rischelp@idt.com)
 *         
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 **************************************************************************
 * Copyright (C) 2000 by Lineo, Inc.
 * Written by Quinn Jensen (jensenq@lineo.com)
 **************************************************************************
 * P. Sadik   Oct 30, 2003
 *
 * Added serial_cons_reset function.
 **************************************************************************
 */

#include <linux/serial_reg.h>

/* set remote gdb baud rate at 115200 */

#define GDB_BAUD 115200
#define CONS_BAUD 9600

extern unsigned int idt_cpu_freq;


/* turn this on to watch the debug protocol echoed on the console port */
#undef DEBUG_REMOTE_DEBUG

#ifdef __MIPSEB__
#define CONS_PORT 0xb8050003u
#define GDB_PORT  0xb8050023u
#else
#define CONS_PORT 0xb8050000u
#define GDB_PORT  0xb8000020u
#endif
           
volatile unsigned char *ports[2] = {
	(volatile unsigned char *)CONS_PORT,
	(volatile unsigned char *)GDB_PORT
};


void reset_gdb_port(void);
void cons_putc(char c);
int port_getc(int port);
void port_putc(int port, char c);

int cons_getc(void)
{
	return port_getc(0);
}

void cons_putc(char c)
{
	port_putc(0, c);
}

void cons_puts(char *s)
{
	while(*s) {
		if(*s == '\n') cons_putc('\r');
		cons_putc(*s);
		s++;
	}
}

void cons_do_putn(int n)
{
	if(n) {
		cons_do_putn(n / 10);
		cons_putc(n % 10 + '0');
	}
}

void cons_putn(int n)
{
	if(n < 0) {
		cons_putc('-');
		n = -n;
	}

	if (n == 0) {
		cons_putc('0');
	} else {
		cons_do_putn(n);
	}
}

#ifdef DEBUG_REMOTE_DEBUG
static enum {HUH, SENDING, GETTING} state;

static void sent(int c)
{
	switch(state) {
	case HUH:
	case GETTING:
		cons_puts("\nSNT ");
		state = SENDING;
		/* fall through */
	case SENDING:
		cons_putc(c);
		break;
	}       
}

static void got(int c)
{
	switch(state) {
	case HUH:
	case SENDING:
		cons_puts("\nGOT ");
		state = GETTING;
		/* fall through */
	case GETTING:
		cons_putc(c);
		break;
	}       
}
#endif /* DEBUG_REMOTE */

static int first = 1;

int getDebugChar(void)
{
	int c;

	if(first) reset_gdb_port();

	c = port_getc(1);

#ifdef DEBUG_REMOTE_DEBUG
	got(c);
#endif

	return c;
}

int port_getc(int p)
{
	volatile unsigned char *port = ports[p];
	int c;

	while((*(port + UART_LSR * 4) & UART_LSR_DR) == 0) {
		continue;
	}       	

	c = *(port + UART_RX * 4);

	return c;
}

void putDebugChar(char c)
{
	if(first) reset_gdb_port();

#ifdef DEBUG_REMOTE_DEBUG
	sent(c);
#endif
	idt_disp_char(3, '%');
	port_putc(1, c);
}

#define OK_TO_XMT (UART_LSR_TEMT | UART_LSR_THRE)

void port_putc(int p, char c)
{
	volatile unsigned char *port = ports[p];
	volatile unsigned char *lsr = port + UART_LSR * 4;

	while((*lsr & OK_TO_XMT) != OK_TO_XMT) {
		continue;
	}

	*(port + UART_TX * 4) = c;
}

void reset_gdb_port(void)
{
	volatile unsigned char *port = ports[1];

	unsigned int DIVISOR = (idt_cpu_freq / 16 / GDB_BAUD);
	first = 0;
#if 1	
	cons_puts("reset_gdb_port: initializing remote debug serial port (internal UART 1, ");
	cons_putn(GDB_BAUD);
	cons_puts("baud, MHz=");
	cons_putn(idt_cpu_freq);
	cons_puts(", divisor=");
	cons_putn(DIVISOR);
	cons_puts(")\n");

	/* reset the port */
	*(port + UART_CSR * 4) = 0;

	/* clear and enable the FIFOs */
	*(port + UART_FCR * 4) = UART_FCR_ENABLE_FIFO | UART_FCR_CLEAR_RCVR | 
		UART_FCR_CLEAR_XMIT | UART_FCR_TRIGGER_14;

	/* set the baud rate */
	*(port + UART_LCR * 4) = UART_LCR_DLAB;		/* enable DLL, DLM registers */
	*(port + UART_DLL * 4) = DIVISOR;
	*(port + UART_DLM * 4) = DIVISOR >> 8;

	/* set the line control stuff and disable DLL, DLM regs */

	*(port + UART_LCR * 4) = UART_LCR_STOP | 	/* 2 stop bits */
		UART_LCR_WLEN8;				/* 8 bit word length */
	
	/* leave interrupts off */
	*(port + UART_IER * 4) = 0;

	/* the modem controls don't leave the chip on this port, so leave them alone */
	*(port + UART_MCR * 4) = 0;
#endif

}

void reset_cons_port(void)
{
	volatile unsigned char *port = ports[0];
	  unsigned int DIVISOR = (idt_cpu_freq / 16 / CONS_BAUD);

	/* reset the port */
	*(port + UART_CSR * 4) = 0;

	/* clear and enable the FIFOs */
	*(port + UART_FCR * 4) = UART_FCR_ENABLE_FIFO | UART_FCR_CLEAR_RCVR | 
		UART_FCR_CLEAR_XMIT | UART_FCR_TRIGGER_14;

	/* set the baud rate */
	*(port + UART_LCR * 4) = UART_LCR_DLAB;		/* enable DLL, DLM registers */

	*(port + UART_DLL * 4) = DIVISOR;
	*(port + UART_DLM * 4) = DIVISOR >> 8;
	/* set the line control stuff and disable DLL, DLM regs */

	*(port + UART_LCR * 4) = UART_LCR_STOP | 	/* 2 stop bits */
		UART_LCR_WLEN8;				/* 8 bit word length */
	
	/* leave interrupts off */
	*(port + UART_IER * 4) = 0;

	/* the modem controls don't leave the chip on this port, so leave them alone */
	*(port + UART_MCR * 4) = 0;
}
